document.addEventListener('DOMContentLoaded', function() {
    // Elementos do Slider
    const slider = document.getElementById('slider');
    const slides = document.querySelectorAll('.slide-link');
    const prevBtn = document.getElementById('prevBtn');
    const nextBtn = document.getElementById('nextBtn');
    const sliderNav = document.getElementById('sliderNav');
    
    // Configurações
    const config = {
        autoPlayDelay: 5000,
        transitionDuration: 500,
        touchThreshold: 50,
        isAutoPlaying: true
    };

    // Estado do Slider
    let currentSlide = 0;
    const totalSlides = slides.length;
    let slideInterval;
    let touchStartX = 0;
    let touchEndX = 0;
    let isTransitioning = false;

    // Inicialização
    function initSlider() {
        createNavigationDots();
        setupTouchEvents();
        setupKeyboardNavigation();
        startAutoPlay();
        updateArrowVisibility();
    }

    // Criar dots de navegação
    function createNavigationDots() {
        slides.forEach((_, index) => {
            const dot = document.createElement('div');
            dot.classList.add('slider-dot');
            if (index === 0) dot.classList.add('active');
            dot.addEventListener('click', () => {
                if (!isTransitioning) goToSlide(index);
            });
            sliderNav.appendChild(dot);
        });
    }

    // Atualizar posição do slider com animação suave
    function updateSliderPosition(instant = false) {
        if (!instant) isTransitioning = true;
        
        slider.style.transition = instant ? 'none' : `transform ${config.transitionDuration}ms ease-in-out`;
        slider.style.transform = `translateX(-${currentSlide * 100}%)`; 

        // Atualizar dots e setas
        updateDots();
        updateArrowVisibility();

        if (!instant) {
            setTimeout(() => {
                isTransitioning = false;
            }, config.transitionDuration);
        }
    }

    // Atualizar dots de navegação
    function updateDots() {
        document.querySelectorAll('.slider-dot').forEach((dot, index) => {
            dot.classList.toggle('active', index === currentSlide);
        });
    }

    // Atualizar visibilidade das setas
    function updateArrowVisibility() {
        prevBtn.style.opacity = currentSlide === 0 ? '0.5' : '1';
        prevBtn.style.cursor = currentSlide === 0 ? 'default' : 'pointer';
        nextBtn.style.opacity = currentSlide === totalSlides - 1 ? '0.5' : '1';
        nextBtn.style.cursor = currentSlide === totalSlides - 1 ? 'default' : 'pointer';
    }

    // Ir para slide específico
    function goToSlide(index) {
        if (index >= 0 && index < totalSlides && !isTransitioning) {
            currentSlide = index;
            updateSliderPosition();
        }
    }

    // Navegação do slider
    function nextSlide() {
    currentSlide = (currentSlide + 1) % totalSlides;
    updateSliderPosition();
}

    function prevSlide() {
    currentSlide = (currentSlide - 1 + totalSlides) % totalSlides;
    updateSliderPosition();
}

    // Configuração de eventos touch
    function setupTouchEvents() {
        slider.addEventListener('touchstart', handleTouchStart, { passive: true });
        slider.addEventListener('touchmove', handleTouchMove, { passive: true });
        slider.addEventListener('touchend', handleTouchEnd);
    }

    function handleTouchStart(e) {
        touchStartX = e.touches[0].clientX;
        pauseAutoPlay();
    }

    function handleTouchMove(e) {
        touchEndX = e.touches[0].clientX;
    }

    function handleTouchEnd() {
        const swipeDistance = touchEndX - touchStartX;
        
        if (Math.abs(swipeDistance) > config.touchThreshold) {
            if (swipeDistance > 0) {
                prevSlide();
            } else {
                nextSlide();
            }
        }
        
        resumeAutoPlay();
    }

    // Navegação por teclado
    function setupKeyboardNavigation() {
        document.addEventListener('keydown', (e) => {
            if (e.key === 'ArrowLeft') {
                prevSlide();
            } else if (e.key === 'ArrowRight') {
                nextSlide();
            }
        });
    }

    // Controle do AutoPlay
    function startAutoPlay() {
        if (config.isAutoPlaying) {
            slideInterval = setInterval(nextSlide, config.autoPlayDelay);
        }
    }

    function pauseAutoPlay() {
        clearInterval(slideInterval);
    }

    function resumeAutoPlay() {
        if (config.isAutoPlaying) {
            startAutoPlay();
        }
    }

    // Event Listeners
    prevBtn.addEventListener('click', (e) => {
        e.preventDefault();
        prevSlide();
    });

    nextBtn.addEventListener('click', (e) => {
        e.preventDefault();
        nextSlide();
    });

    slider.addEventListener('mouseenter', pauseAutoPlay);
    slider.addEventListener('mouseleave', resumeAutoPlay);

    // Listener para final da transição
    slider.addEventListener('transitionend', () => {
        isTransitioning = false;
    });

    // Prevenção de arrasto de imagens
    slider.addEventListener('dragstart', (e) => {
        e.preventDefault();
    });

    // Inicializar o slider
    initSlider();

    // Atualizar slider quando a janela é redimensionada
    let resizeTimeout;
    window.addEventListener('resize', () => {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(() => {
            updateSliderPosition(true);
        }, 250);
    });
});